package edu.uky.ai.lp.logic;

/**
 * A fact is an individual atomic logical statement in function-free predicate
 * logic.
 * 
 * @author Stephen G. Ware
 */
public class Fact implements Expression {
	
	/** The predicate */
	public final String predicate;
	
	/** The terms to which the predicate applies */
	public final Term[] terms;
	
	/**
	 * Constructs a new fact with the given predicate and terms.
	 * 
	 * @param predicate the predicate
	 * @param terms the terms
	 */
	public Fact(String predicate, Term...terms) {
		this.predicate = predicate;
		this.terms = terms;
	}
	
	@Override
	public boolean equals(Object other) {
		if(other instanceof Fact) {
			Fact otherFact = (Fact) other;
			if(!predicate.equals(otherFact.predicate) || terms.length != otherFact.terms.length)
				return false;
			for(int i=0; i<terms.length; i++)
				if(!terms[i].equals(otherFact.terms[i]))
					return false;
			return true;
		}
		return false;
	}
	
	@Override
	public String toString() {
		String str = predicate;
		if(terms.length > 0) {
			String terms = "";
			for(Term term : this.terms)
				terms += "," + term;
			str += "(" + terms.substring(1) + ")";
		}
		return str;
	}
	
	@Override
	public Unifier unify(Formula other, Unifier unifier) {
		if(other instanceof Fact) {
			Fact otherPredication = (Fact) other;
			if(!predicate.equals(otherPredication.predicate) || terms.length != otherPredication.terms.length)
				return null;
			for(int i=0; i<terms.length && unifier != null; i++)
				unifier = terms[i].unify(otherPredication.terms[i], unifier);
			return unifier;
		}
		return null;
	}
	
	@Override
	public Fact substitute(Unifier unifier) {
		Term[] sub = new Term[terms.length];
		for(int i=0; i<terms.length; i++)
			sub[i] = terms[i].substitute(unifier);
		return new Fact(predicate, terms);
	}
}
